using System;
using System.IO;
using DynaPDF;

namespace examples
{
	class Class1
	{
	   private static CPDF m_PDF;

      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write(string.Format("{0}", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage)));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      static int FontNotFoundProc(IntPtr Data, IntPtr PDFFont, IntPtr FontName, TFStyle Style, int StdFontIndex, int IsSymbolFont)
      {
         if (m_PDF.WeightFromStyle(Style) < 500)
         {
            Style  = Style & (TFStyle)0xF;
            Style |= TFStyle.fsRegular;
         }
         return m_PDF.ReplaceFont(PDFFont, "Arial", Style, true);
      }

      static int ReplaceICCProfileProc(IntPtr Data, TICCProfileType Type, int ColorSpace)
      {
         // The most important ICC profiles are available free of charge from Adobe. Just seach for "Adobe icc profiles".
         switch((int)Type)
         {
            case 1:  return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/sRGB.icc");
            case 2:  return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/ISOcoated_v2_bas.ICC"); // This is just an example CMYK profile that can be delivered with DynaPDF.
            default: return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/gray.icc");
         }
      }

      static bool ConvertFile(TConformanceType Type, String InFile, String Invoice, String OutFile)
      {
         int retval;
         TCheckOptions convFlags = TCheckOptions.coDefault_PDFA_3;

         switch (Type)
         {
            case TConformanceType.ctFacturX_Comfort:
            case TConformanceType.ctFacturX_Extended:
            case TConformanceType.ctFacturX_XRechnung: break;
            default:                                   return false;
         }

         m_PDF.CreateNewPDF(null);                         // The output file will be created later
         m_PDF.SetDocInfo(TDocumentInfo.diProducer, null); // No need to override the original producer

         // These flags require some processing time but they are very useful.
         convFlags |= TCheckOptions.coCheckImages | TCheckOptions.coRepairDamagedImages;

         // The flag ifPrepareForPDFA is required. The flag ifImportAsPage makes sure that pages are not converted to templates.
         m_PDF.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage | TImportFlags.ifPrepareForPDFA);
         // The flag if2UseProxy reduces the memory usage.
         m_PDF.SetImportFlags2(TImportFlags2.if2UseProxy);

         if ((retval = m_PDF.OpenImportFile(InFile, TPwdType.ptOpen, null)) < 0)
         {
            if (m_PDF.IsWrongPwd(retval))
            {
               Console.Write("PDFError File is encrypted!");
            }
            m_PDF.FreePDF();
            return false;
         }
         m_PDF.ImportPDFFile(1, 1.0, 1.0);
         m_PDF.CloseImportFile();

         /*
            The invoice should be the first attachment if further files must be attached.
            If the file name of the invoice is not factur-x.xml (case sensitive!) then use AttachFileEx() instead.
            In the case of the German XRechnung the file name must be "xrechnung.xml".
         */

         int ef = m_PDF.AttachFile(Invoice, "EN 16931 compliant invoice", false);
         m_PDF.AssociateEmbFile(TAFDestObject.adCatalog, -1, Type != TConformanceType.ctFacturX_XRechnung ? TAFRelationship.arAlternative : TAFRelationship.arSource, ef);

         // An invoice should not use CMYK colors since a CMYK ICC profile must be embedded in this case and such a profile is pretty large!
         // Note that this code requires the PDF/A Extension for DynaPDF.
         switch ((retval = m_PDF.CheckConformance(Type, convFlags, IntPtr.Zero, FontNotFoundProc, ReplaceICCProfileProc)))
         {
            case 1: m_PDF.AddOutputIntent("../../../../test_files/sRGB.icc");             break;
            case 2: m_PDF.AddOutputIntent("../../../../test_files/ISOcoated_v2_bas.ICC"); break;
            case 3: m_PDF.AddOutputIntent("../../../../test_files/gray.icc");             break;
         }
         // No fatal error occurred?
         if (m_PDF.HaveOpenDoc())
         {
            if (!m_PDF.OpenOutputFile(OutFile))
            {
               m_PDF.FreePDF();
               return false;
            }
            m_PDF.CloseFile();
            return true;
         }
         return false;
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            m_PDF = new CPDF();
            m_PDF.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));

            // Set the license key here if you have one
            // m_PDF.SetLicenseKey("");

            // Non embedded CID fonts depend usually on the availability of external cmaps.
            // So, external cmaps should be loaded if possible.
            m_PDF.SetCMapDir(Path.GetFullPath("../../../../../Resource/CMap"), TLoadCMapFlags.lcmDelayed | TLoadCMapFlags.lcmRecursive);

            String outFile = Directory.GetCurrentDirectory() + "/out.pdf";

            // The profiles Minimum, Basic, and Basic WL are not EN 16931 compliant and hence cannot be used to create e-invoices.
            if (ConvertFile(TConformanceType.ctFacturX_Comfort, "../../../../test_files/test_invoice.pdf", "../../../../test_files/factur-x.xml", outFile))
            {
               Console.Write("PDF file \"{0}\" successfully created!\n", outFile);
               System.Diagnostics.Process p = new System.Diagnostics.Process();
               p.StartInfo.FileName = outFile;
               p.Start();
            }
            m_PDF = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
		}
	}
}
